<?php

// Simple test script for ShareFile connector
// This tests the connector without requiring Composer

echo "ShareFile Connector Test\n";
echo "=======================\n\n";

// Load environment variables manually
function loadEnv($file) {
    if (!file_exists($file)) {
        echo "ERROR: .env file not found. Please copy env.example to .env and configure it.\n";
        return false;
    }
    
    $lines = file($file, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
    foreach ($lines as $line) {
        if (strpos($line, '#') === 0) {
            continue; // Skip comments
        }
        
        $parts = explode('=', $line, 2);
        if (count($parts) === 2) {
            $key = trim($parts[0]);
            $value = trim($parts[1]);
            $_ENV[$key] = $value;
        }
    }
    return true;
}

// Check if .env file exists
if (!loadEnv(__DIR__ . '/.env')) {
    exit(1);
}

// Check required environment variables
$requiredVars = [
    'SHAREFILE_CLIENT_ID',
    'SHAREFILE_CLIENT_SECRET', 
    'SHAREFILE_USERNAME',
    'SHAREFILE_PASSWORD',
    'SHAREFILE_SUBDOMAIN'
];

echo "Checking environment variables...\n";
foreach ($requiredVars as $var) {
    if (empty($_ENV[$var])) {
        echo "ERROR: Missing required environment variable: {$var}\n";
        exit(1);
    }
    echo "✓ {$var} is set\n";
}
echo "\n";

// Check if required files exist
echo "Checking required files...\n";
$requiredFiles = [
    'src/SimpleShareFileAPI.php',
    'src/ClientManager.php'
];

foreach ($requiredFiles as $file) {
    if (!file_exists($file)) {
        echo "ERROR: Required file not found: {$file}\n";
        exit(1);
    }
    echo "✓ {$file} exists\n";
}
echo "\n";

// Test data
$testCompanyData = [
    'coID' => 'TEST' . time(), // Use timestamp to ensure unique ID
    'coName' => 'Test Company ' . date('Y-m-d H:i:s'),
    'firstName' => 'John',
    'lastName' => 'Doe',
    'email' => 'john.doe@testcompany.com'
];

echo "Test company data:\n";
foreach ($testCompanyData as $key => $value) {
    echo "  {$key}: {$value}\n";
}
echo "\n";

// Include required files
require_once __DIR__ . '/src/SimpleShareFileAPI.php';
require_once __DIR__ . '/src/ClientManager.php';

try {
    echo "Initializing ShareFile API...\n";
    $shareFileAPI = new ShareFileConnector\SimpleShareFileAPI();
    echo "✓ ShareFile API initialized successfully\n\n";
    
    echo "Initializing Client Manager...\n";
    $clientManager = new ShareFileConnector\ClientManager($shareFileAPI);
    echo "✓ Client Manager initialized successfully\n\n";
    
    echo "Testing client creation...\n";
    $result = $clientManager->processNewCompany($testCompanyData);
    
    if ($result['success']) {
        echo "✓ SUCCESS: Company created in ShareFile\n";
        echo "  ShareFile Client ID: {$result['clientId']}\n";
        if (isset($result['folderId'])) {
            echo "  ShareFile Folder ID: {$result['folderId']}\n";
        }
    } else {
        echo "⚠ RESULT: {$result['message']}\n";
        if (isset($result['clientId'])) {
            echo "  Existing ShareFile Client ID: {$result['clientId']}\n";
        }
    }
    
} catch (Exception $e) {
    echo "✗ ERROR: " . $e->getMessage() . "\n";
    exit(1);
}

echo "\nTest completed. Check the log file for detailed information.\n"; 