<?php

namespace ShareFileConnector;

use GuzzleHttp\Client;
use GuzzleHttp\Exception\GuzzleException;
use Monolog\Logger;
use Monolog\Handler\StreamHandler;

class ShareFileAPI
{
    private Client $httpClient;
    private Logger $logger;
    private string $accessToken;
    private string $subdomain;
    private string $baseUrl;

    public function __construct()
    {
        $this->logger = new Logger('sharefile_api');
        $this->logger->pushHandler(new StreamHandler($_ENV['LOG_FILE'] ?? 'php://stderr', $_ENV['LOG_LEVEL'] ?? Logger::DEBUG));
        
        $this->subdomain = $_ENV['SHAREFILE_SUBDOMAIN'];
        $this->baseUrl = "https://{$this->subdomain}.sf-api.com/sf/v3";
        
        $this->httpClient = new Client([
            'base_uri' => $this->baseUrl,
            'timeout' => 30,
            'headers' => [
                'Content-Type' => 'application/json',
                'Accept' => 'application/json'
            ]
        ]);
        
        $this->authenticate();
    }

    /**
     * Authenticate with ShareFile using OAuth2
     */
    private function authenticate(): void
    {
        try {
            $response = $this->httpClient->post('https://' . $this->subdomain . '.sharefile.com/oauth/token', [
                'form_params' => [
                    'grant_type' => 'password',
                    'client_id' => $_ENV['SHAREFILE_CLIENT_ID'],
                    'client_secret' => $_ENV['SHAREFILE_CLIENT_SECRET'],
                    'username' => $_ENV['SHAREFILE_USERNAME'],
                    'password' => $_ENV['SHAREFILE_PASSWORD']
                ]
            ]);

            $data = json_decode($response->getBody()->getContents(), true);
            $this->accessToken = $data['access_token'];
            
            $this->logger->info('Successfully authenticated with ShareFile API');
        } catch (GuzzleException $e) {
            $this->logger->error('Authentication failed: ' . $e->getMessage());
            throw new \Exception('Failed to authenticate with ShareFile API: ' . $e->getMessage());
        }
    }

    /**
     * Create a new client in ShareFile
     */
    public function createClient(array $clientData): array
    {
        try {
            $payload = [
                'AccountType' => 'Company',
                'CompanyName' => $clientData['companyName'],
                'FirstName' => $clientData['firstName'],
                'LastName' => $clientData['lastName'],
                'Email' => $clientData['email'],
                'ClientID' => $clientData['clientId'],
                'InternalDisplayName' => $clientData['internalDisplayName']
            ];

            $response = $this->httpClient->post('/Accounts', [
                'headers' => [
                    'Authorization' => 'Bearer ' . $this->accessToken
                ],
                'json' => $payload
            ]);

            $result = json_decode($response->getBody()->getContents(), true);
            $this->logger->info("Successfully created ShareFile client: {$clientData['companyName']} (ID: {$result['Id']})");
            
            return $result;
        } catch (GuzzleException $e) {
            $this->logger->error("Failed to create ShareFile client {$clientData['companyName']}: " . $e->getMessage());
            throw new \Exception("Failed to create ShareFile client: " . $e->getMessage());
        }
    }

    /**
     * Create a folder from template for a client
     */
    public function createFolderFromTemplate(string $clientId, string $folderName, string $templateFolderId): array
    {
        try {
            $payload = [
                'Name' => $folderName,
                'TemplateId' => $templateFolderId,
                'ParentId' => $clientId
            ];

            $response = $this->httpClient->post('/Items', [
                'headers' => [
                    'Authorization' => 'Bearer ' . $this->accessToken
                ],
                'json' => $payload
            ]);

            $result = json_decode($response->getBody()->getContents(), true);
            $this->logger->info("Successfully created folder from template: {$folderName} (ID: {$result['Id']})");
            
            return $result;
        } catch (GuzzleException $e) {
            $this->logger->error("Failed to create folder from template {$folderName}: " . $e->getMessage());
            throw new \Exception("Failed to create folder from template: " . $e->getMessage());
        }
    }

    /**
     * Get client information by ClientID
     */
    public function getClientByClientId(string $clientId): ?array
    {
        try {
            $response = $this->httpClient->get("/Accounts?ClientID={$clientId}", [
                'headers' => [
                    'Authorization' => 'Bearer ' . $this->accessToken
                ]
            ]);

            $data = json_decode($response->getBody()->getContents(), true);
            
            if (!empty($data['value'])) {
                return $data['value'][0];
            }
            
            return null;
        } catch (GuzzleException $e) {
            $this->logger->error("Failed to get client by ClientID {$clientId}: " . $e->getMessage());
            return null;
        }
    }

    /**
     * Update client information
     */
    public function updateClient(string $clientId, array $updateData): array
    {
        try {
            $response = $this->httpClient->patch("/Accounts/{$clientId}", [
                'headers' => [
                    'Authorization' => 'Bearer ' . $this->accessToken
                ],
                'json' => $updateData
            ]);

            $result = json_decode($response->getBody()->getContents(), true);
            $this->logger->info("Successfully updated ShareFile client: {$clientId}");
            
            return $result;
        } catch (GuzzleException $e) {
            $this->logger->error("Failed to update ShareFile client {$clientId}: " . $e->getMessage());
            throw new \Exception("Failed to update ShareFile client: " . $e->getMessage());
        }
    }
} 