<?php

require_once __DIR__ . '/vendor/autoload.php';

use ShareFileConnector\ShareFileAPI;
use ShareFileConnector\ClientManager;
use Dotenv\Dotenv;

// Load environment variables
$dotenv = Dotenv::createImmutable(__DIR__);
$dotenv->load();

/**
 * Script to be called from database trigger
 * Usage: php database_trigger.php <coID>
 */

if ($argc < 2) {
    echo "Usage: php database_trigger.php <coID>\n";
    exit(1);
}

$coID = $argv[1];

try {
    // Get company data from database
    $companyData = getCompanyDataFromDatabase($coID);
    
    if (!$companyData) {
        throw new Exception("Company with coID {$coID} not found in database");
    }

    // Initialize ShareFile API and Client Manager
    $shareFileAPI = new ShareFileAPI();
    $clientManager = new ClientManager($shareFileAPI);

    // Process the new company
    $result = $clientManager->processNewCompany($companyData);

    if ($result['success']) {
        echo "Success: " . $result['message'] . "\n";
        echo "ShareFile Client ID: " . $result['clientId'] . "\n";
        if (isset($result['folderId'])) {
            echo "ShareFile Folder ID: " . $result['folderId'] . "\n";
        }
        exit(0);
    } else {
        echo "Error: " . $result['message'] . "\n";
        exit(1);
    }

} catch (Exception $e) {
    echo "Error: " . $e->getMessage() . "\n";
    exit(1);
}

/**
 * Get company data from eProject database
 * Modify this function according to your database schema
 */
function getCompanyDataFromDatabase(string $coID): ?array
{
    try {
        $pdo = new PDO(
            "sqlsrv:Server=" . $_ENV['DB_HOST'] . ";Database=" . $_ENV['DB_NAME'],
            $_ENV['DB_USER'],
            $_ENV['DB_PASSWORD']
        );
        
        $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

        // Adjust this query based on your actual database schema
        $sql = "
            SELECT 
                c.coID,
                c.coName,
                p.firstName,
                p.lastName,
                p.email
            FROM Companies c
            LEFT JOIN PrimaryContacts p ON c.coID = p.coID
            WHERE c.coID = :coID
        ";

        $stmt = $pdo->prepare($sql);
        $stmt->execute(['coID' => $coID]);
        
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if ($result) {
            return [
                'coID' => $result['coID'],
                'coName' => $result['coName'],
                'firstName' => $result['firstName'],
                'lastName' => $result['lastName'],
                'email' => $result['email']
            ];
        }
        
        return null;
        
    } catch (PDOException $e) {
        throw new Exception("Database error: " . $e->getMessage());
    }
} 