# Simple ShareFile Connector Setup

This guide will help you set up the ShareFile connector without requiring Composer or external dependencies.

## Quick Setup

### Step 1: Configure Environment Variables

1. Copy the environment template:
   ```bash
   cp env.example .env
   ```

2. Edit the `.env` file with your ShareFile API credentials:
   ```env
   # ShareFile API Configuration
   SHAREFILE_CLIENT_ID=your_client_id_here
   SHAREFILE_CLIENT_SECRET=your_client_secret_here
   SHAREFILE_USERNAME=your_username_here
   SHAREFILE_PASSWORD=your_password_here
   SHAREFILE_SUBDOMAIN=your_subdomain_here
   
   # Logging Configuration
   LOG_FILE=logs/sharefile_connector.log
   
   # Template Configuration (optional)
   SHAREFILE_TEMPLATE_FOLDER_ID=your_template_folder_id_here
   ```

### Step 2: Get ShareFile API Credentials

1. Log into your ShareFile account
2. Go to **Admin** → **API**
3. Create a new API application
4. Note down:
   - **Client ID**
   - **Client Secret**
   - **Subdomain** (the part before `.sharefile.com` in your ShareFile URL)

### Step 3: Test the Connection

Run the test script to verify everything works:

```bash
php test_simple.php
```

This will:
- Check your environment variables
- Test the ShareFile API connection
- Create a test client in ShareFile
- Show you the results

### Step 4: Upload to Your Server

Upload these files to your Lucee server:
- `simple_webhook.php` (main webhook endpoint)
- `src/SimpleShareFileAPI.php` (API client)
- `src/ClientManager.php` (business logic)
- `.env` (your configuration)

## Integration Options

### Option 1: Webhook Integration (Recommended)

Use `simple_webhook.php` as your webhook endpoint. Add this to your Lucee company save logic:

```cfml
<cfscript>
// After saving company
var shareFileData = {
    "coID": company.coID,
    "coName": company.coName,
    "firstName": company.primaryContact.firstName,
    "lastName": company.primaryContact.lastName,
    "email": company.primaryContact.email
};

// Call webhook
var httpService = new http();
httpService.setMethod("POST");
httpService.setUrl("https://yourserver.com/path/to/simple_webhook.php");
httpService.addParam(type="header", name="Content-Type", value="application/json");
httpService.addParam(type="body", value=serializeJSON(shareFileData));

var result = httpService.send().getPrefix();
</cfscript>
```

### Option 2: Direct PHP Integration

Include the connector directly in your PHP code:

```php
<?php
require_once 'src/SimpleShareFileAPI.php';
require_once 'src/ClientManager.php';

// Load environment variables
function loadEnv($file) {
    $lines = file($file, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
    foreach ($lines as $line) {
        if (strpos($line, '#') === 0) continue;
        $parts = explode('=', $line, 2);
        if (count($parts) === 2) {
            $_ENV[trim($parts[0])] = trim($parts[1]);
        }
    }
}
loadEnv('.env');

// Use the connector
$shareFileAPI = new ShareFileConnector\SimpleShareFileAPI();
$clientManager = new ShareFileConnector\ClientManager($shareFileAPI);

$companyData = [
    'coID' => '12345',
    'coName' => 'Test Company',
    'firstName' => 'John',
    'lastName' => 'Doe',
    'email' => 'john.doe@testcompany.com'
];

$result = $clientManager->processNewCompany($companyData);
?>
```

## Testing

### Test the Webhook

```bash
curl -X POST https://yourserver.com/path/to/simple_webhook.php \
  -H "Content-Type: application/json" \
  -d '{
    "coID": "12345",
    "coName": "Test Company",
    "firstName": "John",
    "lastName": "Doe",
    "email": "john.doe@testcompany.com"
  }'
```

### Check Logs

Monitor the integration by checking the log file:
```bash
tail -f logs/sharefile_connector.log
```

## Troubleshooting

### Common Issues

1. **"Authentication failed"**
   - Verify your ShareFile API credentials
   - Check your subdomain
   - Ensure your API application has correct permissions

2. **"Missing required field"**
   - Make sure all required fields are provided
   - Check the data format in your Lucee integration

3. **"Client already exists"**
   - This is normal if you test with the same coID
   - The connector will return the existing client ID

4. **"cURL error"**
   - Ensure cURL is enabled in your PHP installation
   - Check your server's internet connectivity

### Debug Mode

To get more detailed information, check the log file at `logs/sharefile_connector.log`. All operations are logged with timestamps.

## File Structure

```
ShareFile-api/
├── .env                          # Your configuration (create this)
├── env.example                   # Configuration template
├── simple_webhook.php            # Webhook endpoint (use this)
├── test_simple.php               # Test script
├── src/
│   ├── SimpleShareFileAPI.php    # API client (no dependencies)
│   └── ClientManager.php         # Business logic
└── logs/                         # Log files (created automatically)
    └── sharefile_connector.log
```

## Next Steps

1. **Configure your ShareFile API credentials**
2. **Test the connection** with `test_simple.php`
3. **Integrate with your Lucee application**
4. **Set up your template folder** (optional)
5. **Monitor the logs** for any issues

## Support

If you encounter issues:
1. Check the log file for detailed error messages
2. Verify your ShareFile API credentials
3. Test with the provided test script
4. Review the ShareFile API documentation at api.sharefile.com/docs 